<?php
// upload_revision.php
session_start();
require 'db_connection.php'; 

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die("Invalid request method.");
}

// 1. Grab the token and the uploaded file
$access_token = htmlspecialchars(trim($_POST['access_token']));
$uploaded_file = $_FILES['revision_file'];

if (!isset($uploaded_file) || $uploaded_file['error'] !== UPLOAD_ERR_OK) {
    die("Error: Please select a valid file to upload.");
}

// 2. Validate the Magic Link and current Status
// We ONLY allow uploads if the status is exactly 'downloaded'
$stmt = $pdo->prepare("
    SELECT share_id, document_id, recipient_name, recipient_email 
    FROM document_shares 
    WHERE access_token = ? AND status = 'downloaded'
");
$stmt->execute([$access_token]);
$share_data = $stmt->fetch();

if (!$share_data) {
    die("Error: Invalid link, or you have already submitted your revision. Access revoked.");
}

// 3. Strict Security: File Type Validation
// We only want Word or PowerPoint files back. No executables (.exe) or PHP scripts.
$allowed_mimes = [
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document', // .docx
    'application/vnd.openxmlformats-officedocument.presentationml.presentation' // .pptx
];

$file_mime = mime_content_type($uploaded_file['tmp_name']);
if (!in_array($file_mime, $allowed_mimes)) {
    die("Security Error: Only .docx or .pptx files are allowed.");
}

// 4. Secure Storage Setup
// Store revisions in a directory OUTSIDE your public HTML folder so they can't be guessed
$upload_directory = '/var/www/ncdc_secure/revisions/'; 
if (!is_dir($upload_directory)) {
    mkdir($upload_directory, 0755, true);
}

// Create a unique, randomized filename to prevent overwriting
$new_filename = 'REV_' . $share_data['share_id'] . '_' . bin2hex(random_bytes(8)) . '.docx';
$destination = $upload_directory . $new_filename;

// 5. Move the file and Lock the Account
if (move_uploaded_file($uploaded_file['tmp_name'], $destination)) {
    
    // THE LOCKOUT: Change status to 'returned' and save the file path
    $update_stmt = $pdo->prepare("
        UPDATE document_shares 
        SET status = 'returned', returned_file_path = ?, updated_at = NOW() 
        WHERE share_id = ?
    ");
    $update_stmt->execute([$destination, $share_data['share_id']]);

    // Log the successful secure upload
    $ip = $_SERVER['REMOTE_ADDR'];
    $pdo->prepare("
        INSERT INTO audit_logs (guest_email, document_id, action, ip_address) 
        VALUES (?, ?, 'Revision_Uploaded_And_Locked', ?)
    ")->execute([$share_data['recipient_email'], $share_data['document_id'], $ip]);

    // Success Message
    echo "<h2>Upload Successful.</h2>";
    echo "<p>Thank you, " . htmlspecialchars($share_data['recipient_name']) . ".</p>";
    echo "<p>Your revised document has been securely transmitted to NCDC. <strong>Your access link has now been permanently deactivated.</strong></p>";

} else {
    die("System Error: Could not save the uploaded file. Please contact NCDC IT.");
}
?>